<?php

namespace App\Console\Commands;

use Botble\Ecommerce\Models\ProductCategory;
use Botble\Media\Facades\RvMedia;
use Botble\Media\Models\MediaFile;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;

class MigrateCategoryPdfs extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'category:migrate-pdfs
                            {--dry-run : Show what would be migrated without actually doing it}
                            {--force : Force migration even if PDF already exists}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Migrate existing PDFs from public/storage/downloads to categories';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🔄 Starting Category PDF Migration...');

        $downloadsPath = public_path('storage/downloads');
        $isDryRun = $this->option('dry-run');
        $isForce = $this->option('force');

        if (!File::exists($downloadsPath)) {
            $this->error("❌ Downloads directory not found: {$downloadsPath}");
            return 1;
        }

        $pdfFiles = File::glob($downloadsPath . '/*.pdf');

        if (empty($pdfFiles)) {
            $this->warn('⚠️  No PDF files found in downloads directory');
            return 0;
        }

        $this->info("📁 Found " . count($pdfFiles) . " PDF files");

        $categories = ProductCategory::all();
        $migrated = 0;
        $skipped = 0;
        $matched = 0;

        foreach ($pdfFiles as $pdfPath) {
            $fileName = pathinfo($pdfPath, PATHINFO_FILENAME);
            $this->line("🔍 Processing: {$fileName}.pdf");

            // Try to find matching category
            $matchedCategory = $this->findMatchingCategory($categories, $fileName);

            if (!$matchedCategory) {
                $this->warn("   ⚠️  No matching category found for: {$fileName}");
                $skipped++;
                continue;
            }

            $matched++;
            $this->info("   ✅ Matched with category: {$matchedCategory->name}");

            // Check if category already has a PDF
            if ($matchedCategory->pdf_file && !$isForce) {
                $this->warn("   ⚠️  Category already has a PDF (use --force to override)");
                $skipped++;
                continue;
            }

            if ($isDryRun) {
                $this->line("   🔮 [DRY RUN] Would upload and assign to category");
                continue;
            }

            // Upload to media manager
            try {
                $uploadedFile = $this->uploadToMediaManager($pdfPath, $fileName);

                if ($uploadedFile) {
                    // Update category
                    $matchedCategory->update([
                        'pdf_file' => $uploadedFile->url,
                        'pdf_title' => $matchedCategory->name . ' Catalog',
                        'pdf_description' => 'Download our comprehensive ' . $matchedCategory->name . ' catalog',
                    ]);

                    $this->info("   🎉 Successfully migrated!");
                    $migrated++;
                } else {
                    $this->error("   ❌ Failed to upload file");
                    $skipped++;
                }
            } catch (\Exception $e) {
                $this->error("   ❌ Error: " . $e->getMessage());
                $skipped++;
            }
        }

        $this->newLine();
        $this->info("📊 Migration Summary:");
        $this->table(['Metric', 'Count'], [
            ['Total PDFs Found', count($pdfFiles)],
            ['Categories Matched', $matched],
            ['Successfully Migrated', $migrated],
            ['Skipped', $skipped],
        ]);

        if ($isDryRun) {
            $this->info("🔮 This was a dry run. Use the command without --dry-run to perform actual migration.");
        }

        return 0;
    }

    private function findMatchingCategory($categories, $fileName)
    {
        // Direct name match
        foreach ($categories as $category) {
            if (Str::slug($category->name) === Str::slug($fileName)) {
                return $category;
            }
        }

        // Partial match
        foreach ($categories as $category) {
            $categorySlug = Str::slug($category->name);
            $fileSlug = Str::slug($fileName);

            if (Str::contains($categorySlug, $fileSlug) || Str::contains($fileSlug, $categorySlug)) {
                return $category;
            }
        }

        // Fuzzy matching for common variations
        $fileSlug = Str::slug($fileName);
        foreach ($categories as $category) {
            $categorySlug = Str::slug($category->name);

            // Handle plurals, spaces, and common variations
            if ($this->isFuzzyMatch($categorySlug, $fileSlug)) {
                return $category;
            }
        }

        return null;
    }

    private function isFuzzyMatch($categorySlug, $fileSlug)
    {
        // Remove common words and variations
        $commonWords = ['safe', 'safes', 'cabinet', 'cabinets', 'box', 'boxes'];

        foreach ($commonWords as $word) {
            $categorySlug = str_replace($word, '', $categorySlug);
            $fileSlug = str_replace($word, '', $fileSlug);
        }

        // Check similarity
        similar_text($categorySlug, $fileSlug, $percentage);

        return $percentage > 70; // 70% similarity threshold
    }

    private function uploadToMediaManager($filePath, $fileName)
    {
        try {
            // Create a temporary uploaded file object
            $file = new \Illuminate\Http\UploadedFile(
                $filePath,
                $fileName . '.pdf',
                'application/pdf',
                null,
                true
            );

            // Upload using RvMedia
            $result = RvMedia::handleUpload($file, 0);

            if ($result['error']) {
                throw new \Exception($result['message']);
            }

            return MediaFile::find($result['data']->id);
        } catch (\Exception $e) {
            $this->error("Upload failed: " . $e->getMessage());
            return null;
        }
    }
}
