# Category PDF Downloads Feature

## Overview
This feature allows administrators to upload PDF files for product categories and display download buttons on category pages across all Shofy theme variants.

## Features
- ✅ **Admin Management**: Upload, edit, and manage PDF files for categories
- ✅ **Multi-theme Support**: Works across all Shofy theme variants (shofy, shofy-fashion, shofy-beauty, shofy-grocery, shofy-jewelry)
- ✅ **Conditional Display**: PDF download section only appears when a PDF is uploaded
- ✅ **Responsive Design**: Mobile-friendly download cards
- ✅ **Scalable Architecture**: Easy to extend and maintain
- ✅ **Migration Tool**: Command to migrate existing PDFs from public/storage/downloads

## Database Schema
New fields added to `ec_product_categories` table:
- `pdf_file` (string, nullable) - Path to the PDF file in media manager
- `pdf_title` (string, nullable) - Custom title for download button
- `pdf_description` (text, nullable) - Optional description for the PDF

## Admin Interface
Navigate to **Admin → Ecommerce → Product Categories → Create/Edit Category**

New PDF section includes:
- **PDF File**: Upload field for PDF files
- **PDF Title**: Custom title (defaults to "Category Name Catalog")
- **PDF Description**: Optional description text

## Frontend Display
The PDF download section appears:
- **Position**: After category name, before products listing
- **Visibility**: Only when PDF is uploaded for the category
- **Style**: Responsive card with PDF icon, title, description, and download button

## Migration Tool
Migrate existing PDFs from `public/storage/downloads/` to categories:

```bash
# Dry run (preview what would be migrated)
php artisan category:migrate-pdfs --dry-run

# Actual migration
php artisan category:migrate-pdfs

# Force migration (overwrite existing PDFs)
php artisan category:migrate-pdfs --force
```

The migration tool:
- Automatically matches PDF filenames to category names
- Uses fuzzy matching for variations in naming
- Uploads PDFs to media manager
- Associates them with matching categories
- Provides detailed migration report

## File Locations

### Backend Files
- `database/migrations/*_add_pdf_fields_to_product_categories_table.php`
- `platform/plugins/ecommerce/src/Models/ProductCategory.php`
- `platform/plugins/ecommerce/src/Forms/ProductCategoryForm.php`
- `platform/plugins/ecommerce/src/Http/Requests/ProductCategoryRequest.php`
- `platform/plugins/ecommerce/src/Tables/ProductCategoryTable.php`
- `platform/plugins/ecommerce/src/Services/HandleFrontPages.php`

### Frontend Files
- `platform/plugins/ecommerce/resources/views/themes/includes/product-listing-page-description.blade.php`
- `platform/themes/shofy/partials/category-pdf-download.blade.php`
- `platform/themes/shofy-fashion/partials/category-pdf-download.blade.php`
- `platform/themes/shofy-beauty/partials/category-pdf-download.blade.php`
- `platform/themes/shofy-grocery/partials/category-pdf-download.blade.php`
- `platform/themes/shofy-jewelry/partials/category-pdf-download.blade.php`

### Language Files
- `platform/plugins/ecommerce/resources/lang/en/product-categories.php`

### Commands
- `app/Console/Commands/MigrateCategoryPdfs.php`

## Model Methods
New methods in `ProductCategory` model:

```php
// Check if category has a PDF
$category->hasPdf(): bool

// Get PDF download URL
$category->getPdfUrl(): ?string

// Get PDF title (custom or default)
$category->getPdfTitle(): string

// Get PDF description
$category->getPdfDescription(): ?string
```

## Template Usage
In Blade templates:

```blade
@if ($category->hasPdf())
    @include(Theme::getThemeNamespace('partials.category-pdf-download'), ['category' => $category])
@endif
```

## Styling
The PDF download section includes:
- Responsive flexbox layout
- PDF icon with red color
- Hover effects
- Mobile-responsive design
- Bootstrap-compatible styling

## Language Support
All text is translatable through language files:
- `plugins/ecommerce::product-categories.pdf.download`
- `plugins/ecommerce::product-categories.pdf.catalog`
- `plugins/ecommerce::product-categories.pdf.file`
- etc.

## Security Considerations
- PDF files are uploaded through Botble's media manager
- Files are validated for PDF format
- Access is controlled through category visibility
- Download links use media manager URLs

## Performance
- Conditional rendering - no impact when no PDF exists
- Minimal database overhead (3 additional nullable fields)
- CSS is inline for optimal loading
- No additional HTTP requests for checking PDF existence

## Browser Support
- Modern browsers with CSS flexbox support
- Mobile responsive design
- Progressive enhancement approach

## Future Enhancements
Potential additions:
- Multiple PDF support per category
- PDF version control
- Download tracking/analytics
- PDF preview functionality
- Bulk PDF upload interface
